// Code to merge a sequence of GMV files created with parpp3d++
// for a parallel run with several processors to one single GMV file.
//
// History:
// 2004/05/25   v1.2.0   Ported to g++ 3.4.0, now it works on:
//                       * Linux 2.4.21-215-athlon (stingray), g++ 3.4.0 & g++ 3.3.x
//                       * Linux 2.4.23 (hyperion),            g++ 3.2.2 & g++ 2.95.4
//                       * Linux 2.4.18 (odin/helics),         g++ 2.95.4
//                       * SunOS 5.9 (teroknor),               g++ 3.3.3 & Sun C++ 5.5 Patch 113817-07
//                       * Hitachi SR8000 (LRZ M),             g++ 2.95.2 & KCC 3.4
//                       * IBM Regatta 4 (jump),               KCC 4.0 & KCC 3.4 & g++ 2.9-aix51-020209

#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#define FILE_NOT_FOUND             -1
#define NOT_AVS_FORMAT             -2
#define NODE_INFO_NOT_FOUND        -3
#define ELEMENT_INFO_NOT_FOUND     -4
#define VELOCITY_INFO_NOT_FOUND    -5
#define PRESSURE_INFO_NOT_FOUND    -6
#define TEMPERATURE_INFO_NOT_FOUND -7
#define ONE_FILE_WITHOUT_VELOCITY_INFO         -8
#define ONE_FILE_WITHOUT_PRESSURE_INFO         -9
#define ONE_FILE_WITHOUT_TEMPERATURE_INFO     -10
#define ONE_FILE_WITHOUT_VORTICITY_X_INFO     -11
#define ONE_FILE_WITHOUT_VORTICITY_Y_INFO     -12
#define ONE_FILE_WITHOUT_VORTICITY_Z_INFO     -13
#define ONE_FILE_WITHOUT_VORTICITY_TOTAL_INFO -14

using std::string;
using std::vector;
using std::cout;
using std::cerr;

#define VERSION "v1.2.0"

string int_to_string (const unsigned int i, const unsigned int digits) {
    string s;
    switch (digits) 
    {
    case 4:
        s += '0' + i/1000;
    case 3:
        s += '0' + (i%1000)/100;
    case 2:
        s += '0' + (i%100)/10;
    case 1:
        s += '0' + i%10;
        break;
    default:
        s += "invalid digits information";
    };
    return s;
}


int main(int argc,char* argv[])
{
    string infile, outfile;
    string entry;
    int    k;                                     // Schleifenzaehler
    int    vInitialTimeStep  = 0, vNumTimeSteps = 0;
    int    vNumProcs         = 0;
    int    vNumVerticesTotal = 0, vNumElementsTotal = 0;
    bool   flag;
    int    vNumVerticesCurrentFile = 0, vNumVerticesSoFar = 0; // #vertices in current input file and all that have been read
    int    vNumElementsCurrentFile = 0, vNumElementsSoFar = 0; // dito for elements
    bool   flagVelocity    = false;
    bool   flagPressure    = false;
    bool   flagTemperature = false;
    bool   flagVorticity_x = false;
    bool   flagVorticity_y = false;
    bool   flagVorticity_z = false;
    bool   flagVorticity_total = false;
    bool   flagTime        = false;

    // -----------
    // Print Usage
    // -----------
    if (argc != 5) {
        cout << "Usage: " << argv[0] << " <filename_prefix>\n"
	     << "          <initial time step> <#time steps> <#files per time step>\n\n"
	     << "Merges <#files> avs  input files  from a pp3d++ run to one\n"
	     << "single avs file.  The format of those files that are to be\n"
	     << "merged is expected to be\n"
	     << "  <filename_prefix>.t001.p000.inp,\n"
	     << "  <filename_prefix>.t001.p001.inp\n"
	     << "A file named <filename_prefix>.inp is created. It contains\n"
	     << "besides all the information  from the input files material\n"
	     << "characteristics: the cells from different input files have\n"
	     << "different values for their material characteristics.\n\n"
// 	     << "A set of inp input  files that contain only grid information\n"
// 	     << "can be merged by setting the initial time step to -1 and the\n"
// 	     << "number of time steps to 1.  In this case the file pattern is\n"
// 	     << "exptected be\n"
// 	     << "  <filename_prefix>.p000.inp,\n"
// 	     << "  <filename_prefix>.p001.inp\n\n"
	     << "Version: " << VERSION << "\n\n";
        exit (0);
    }

    vInitialTimeStep = atoi(argv[2]);             // initial timestep
    vNumTimeSteps = atoi(argv[3]);                // number of timesteps
    vNumProcs     = atoi(argv[4]);                // number of input files per timestep
    vector< int > vNumElements(vNumProcs, 0);     // remember number of elements for each separate input file

    cout << argv[0] << " " << VERSION << '\n';
    for (int t=vInitialTimeStep;
	 abs(t - vInitialTimeStep) < abs(vNumTimeSteps);
	 t += vNumTimeSteps / abs(vNumTimeSteps) * 1) {
        // Re-initialize counters for each file
        vNumVerticesTotal = 0;       vNumElementsTotal = 0;
        vNumVerticesCurrentFile = 0; vNumVerticesSoFar = 0;
        vNumElementsCurrentFile = 0; vNumElementsSoFar = 0;

        // ----------------------------------------------------------
        // Open all files and sum the number of vertices and elements.
        // ----------------------------------------------------------
        for (int i=0; i < vNumProcs; i++) {
            infile = argv[1];

	    // Build up the current file name string
//  	    if (vInitialTimeStep > -1) {
		infile += ".t" ;
		if (t < 1000) {
		    infile += int_to_string (t, 3);
		} else {
		    infile += int_to_string (t, 4);
		}
//  	    }
            infile += ".p" ;
            infile += int_to_string (i, 3);
            infile += ".inp";

	    // Tell the user what we're doing right now.
            cout << "Reading input file <" << infile << "> ... ";
	    cout.flush();

            // Open file
            std::ifstream *file = new std::ifstream(infile.c_str());
            if (! *file) {
                cerr << '\n' << argv[0] << ": "
		     << "Cannot open file <" << infile << ">\n"
		     << "Program aborted. Nothing done.\n\n";
                exit (FILE_NOT_FOUND);
            }

            // Read number of vertices
            *file >> entry;
            vNumVerticesTotal += atoi(entry.c_str());

            // Read number of elements
            *file >> entry;
            vNumElements[i]    = atoi(entry.c_str());
            vNumElementsTotal += atoi(entry.c_str());

            // Close file.
            delete file;
            cout << "done.\n";
        } // end for loop over all input files

        // -----------------
        // Print statistics.
        // -----------------
        cout << '\n'
	     << "Statistics:\n"
	     << "-----------\n"
	     << "Number of vertices: " << vNumVerticesTotal << '\n'
	     << "Number of elements: " << vNumElementsTotal << "\n\n";


        // ----------------
        // Allocate memory.
        // ----------------
        vector< vector<double> > vertex     (vNumVerticesTotal, vector<double> (3, -999.999));
        vector< vector<int>    > cell       (vNumElementsTotal, vector<int>    (8, -999));
        vector< vector<double> > velocity   (vNumVerticesTotal, vector<double> (3, -999.999));
        vector<double>           pressure   (vNumVerticesTotal, -999.999);
        vector<double>           temperature(vNumVerticesTotal, -999.999);
	string              data_description;

        // -------------------------
        // Collect distributed data.
        // -------------------------
        for (int i=0; i < vNumProcs; i++) {
            infile = argv[1];
//  	    if (vInitialTimeStep > -1) {
		infile += ".t" ;
		if (t < 1000) {
		    infile += int_to_string (t, 3);
		} else {
		    infile += int_to_string (t, 4);
		}
//  	    }
            infile += ".p" ;
            infile += int_to_string (i, 3);
            infile += ".inp";

            cout << "Processing input file <" << infile << "> ... ";
	    cout.flush();

            // Open file
            std::ifstream *file = new std::ifstream(infile.c_str());
            if (! *file) {
                cerr << '\n' << argv[0] << ": "
		     << "Cannot open file <" << infile << ">\n"
		     << "Program aborted. Nothing done.\n\n";
                exit (FILE_NOT_FOUND);
            }

            // Read number of vertices and the coordinates of those vertices
            *file >> vNumVerticesCurrentFile;
            *file >> vNumElementsCurrentFile;
	    *file >> entry; *file >> entry; *file >> entry;     // Skip " 5 0 0"
            for (int j=0; j < vNumVerticesCurrentFile; j++) {
		*file >> entry;			                // Skip number of vertex
                *file >> vertex[vNumVerticesSoFar + j][0];      // read x component of vertex j
                *file >> vertex[vNumVerticesSoFar + j][1];      // read y component of vertex j
                *file >> vertex[vNumVerticesSoFar + j][2];      // read z component of vertex j
	    }

            // Read cell info
            for (int j=0; j < vNumElementsCurrentFile; j++) {
                *file >> entry;                       // Skip number of cells
                *file >> entry;                       // Skip "1"
                *file >> entry;                       // Skip "hex"
                *file >> entry;                       // Edge 1
                cell[vNumElementsSoFar + j][0] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 2
                cell[vNumElementsSoFar + j][1] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 3
                cell[vNumElementsSoFar + j][2] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 4
                cell[vNumElementsSoFar + j][3] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 5
                cell[vNumElementsSoFar + j][4] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 6
                cell[vNumElementsSoFar + j][5] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 7
                cell[vNumElementsSoFar + j][6] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 8
                cell[vNumElementsSoFar + j][7] = vNumVerticesSoFar + atoi(entry.c_str());
            }

            // Read data info
	    // Skip next 6 lines
//  	    for (unsigned int j = 1;  j <= 6;  j++) {
//  		*file >> entry;
//  	    }
	    for (unsigned int j = 0;  j <= 6;  j++) 
		while (file->get() != '\n') ;
	    
  	    // Read velocity in each vertex
	    for (int j=0; j < vNumVerticesCurrentFile; j++) {
		*file >> entry;			  // Skip number of vertex
		*file >> velocity[vNumVerticesSoFar + j][0];      // read x component of velocity in vertex j
		*file >> velocity[vNumVerticesSoFar + j][1];      // read y component of velocity in vertex j
		*file >> velocity[vNumVerticesSoFar + j][2];      // read z component of velocity in vertex j
		*file >> pressure[vNumVerticesSoFar + j];         // read pressure in vertex j
		*file >> temperature[vNumVerticesSoFar + j];      // read temperature in vertex j
	    }

            // Update log variables
            vNumVerticesSoFar += vNumVerticesCurrentFile;
            vNumElementsSoFar += vNumElementsCurrentFile;

            // Close file.
            delete file;
            cout << " done.\n";
        } // end for loop over all input files


        // ----------------------------
        // Write data into one big file
        // ----------------------------
        outfile = argv[1];
	if (vInitialTimeStep > -1) {
	    outfile += ".t" ;
	    if (t < 1000) {
		outfile += int_to_string (t, 3);
	    } else {
		outfile += int_to_string (t, 4);
	    }
	}
        outfile += ".inp";

        cout << "Writing output file <" << outfile << "> ... ";
	cout.flush();

        // Open file
        std::ofstream *file = new std::ofstream(outfile.c_str());
        if (! *file) {
            cerr << '\n' << argv[0] << ": "
		 << "Cannot open file <" << outfile << ">\n"
		 << "Program aborted. Nothing done.\n\n";
            exit (FILE_NOT_FOUND);
        }

        // Write data to file
	// vertices, elements, number of fields, ?, ?
        *file << vNumVerticesTotal << " " << vNumElementsTotal << " 5 0 0\n";
	k = 0;
	// Write node data
	for (vector< vector<double> >::const_iterator j = vertex.begin(); j < vertex.end(); ++j) {
	    k++;
	    *file 
		<< k << ' '
		<< j[0][0] << ' ' << j[0][1] << ' ' << j[0][2] << "\n"; // Writing x, y, z coordinates
	}

	// Write cell data
	k = 0;
	for (vector< vector<int> >::const_iterator j = cell.begin(); j < cell.end(); ++j) {
	    k++;
	    *file << k << " 1 hex " 
		  << j[0][0] << ' ' << j[0][1] << ' ' << j[0][2] << ' ' << j[0][3] << ' '
		  << j[0][4] << ' ' << j[0][5] << ' ' << j[0][6] << ' ' << j[0][7] << '\n';
	}

	// Write field information
	*file 
	    << "5 1 1 1 1 1\n"
	    << "v1 , m / s\n"
	    << "v2 , m / s\n"
	    << "v3 , m / s\n"
	    << "p , pa\n"
	    << "Temp , Kelvin\n";

            // Write velocity, pressure and temperature data for each node
	k = 0;
	for (vector< vector<double> >::const_iterator j = vertex.begin(); j < vertex.end(); ++j) {
	    *file 
		<< k+1 << ' '
		<< velocity[k][0] << ' ' << velocity[k][1] << ' ' << velocity[k][2] << ' '
		<< pressure[k] << ' ' << temperature[k] << '\n';
	    k++;
	}

        // Close file
        delete file;
        cout << "done.\n";
    } // for each timestep

    return 0;
}

