// Code to merge a sequence of GMV files created with parpp3d++
// for a parallel run with several processors to one single GMV file.
//
// History:
// 2004/06/03   v1.3.0   Included support for vorticity field in input data,
//                       Existence of the parameters temperature, vorticity etc.
//                       is now optional and not mandatory.
// 2004/05/25   v1.2.0   Ported to g++ 3.4.0, now it works on:
//                       * Linux 2.4.21-215-athlon (stingray), g++ 3.4.0 & g++ 3.3.x
//                       * Linux 2.4.23 (hyperion),            g++ 3.2.2 & g++ 2.95.4
//                       * Linux 2.4.18 (odin/helics),         g++ 2.95.4
//                       * SunOS 5.9 (teroknor),               g++ 3.3.3 & Sun C++ 5.5 Patch 113817-07
//                       * Hitachi SR8000 (LRZ M),             g++ 2.95.2 & KCC 3.4
//                       * IBM Regatta 4 (jump),               KCC 4.0 & KCC 3.4 & g++ 2.9-aix51-020209

#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#define FILE_NOT_FOUND             -1
#define NOT_GMV_FORMAT             -2
#define NODE_INFO_NOT_FOUND        -3
#define ELEMENT_INFO_NOT_FOUND     -4
#define VELOCITY_INFO_NOT_FOUND    -5
#define PRESSURE_INFO_NOT_FOUND    -6
#define TEMPERATURE_INFO_NOT_FOUND -7
#define ONE_FILE_WITHOUT_VELOCITY_INFO         -8
#define ONE_FILE_WITHOUT_PRESSURE_INFO         -9
#define ONE_FILE_WITHOUT_TEMPERATURE_INFO     -10
#define ONE_FILE_WITHOUT_VORTICITY_X_INFO     -11
#define ONE_FILE_WITHOUT_VORTICITY_Y_INFO     -12
#define ONE_FILE_WITHOUT_VORTICITY_Z_INFO     -13
#define ONE_FILE_WITHOUT_VORTICITY_TOTAL_INFO -14

using std::string;
using std::vector;
using std::cout;
using std::cerr;

#define VERSION "v1.3.0"

string int_to_string (const unsigned int i, const unsigned int digits) {
    string s;
    switch (digits) 
    {
    case 4:
        s += '0' + i/1000;
    case 3:
        s += '0' + (i%1000)/100;
    case 2:
        s += '0' + (i%100)/10;
    case 1:
        s += '0' + i%10;
        break;
    default:
        s += "invalid digits information";
    };
    return s;
}


int main(int argc,char* argv[])
{
    string infile, outfile;
    string entry;
    int    k;                                     // Schleifenzaehler
    int    vInitialTimeStep  = 0, vNumTimeSteps = 0;
    int    vNumProcs         = 0;
    int    vNumVerticesTotal = 0, vNumElementsTotal = 0;
    bool   flag;
    int    vNumVerticesCurrentFile = 0, vNumVerticesSoFar = 0; // #vertices in current input file and all that have been read
    int    vNumElementsCurrentFile = 0, vNumElementsSoFar = 0; // dito for elements
    bool   flagVelocity    = false;
    bool   flagPressure    = false;
    bool   flagTemperature = false;
    bool   flagVorticity_x = false;
    bool   flagVorticity_y = false;
    bool   flagVorticity_z = false;
    bool   flagVorticity_total = false;
    bool   flagTime        = false;

    // -----------
    // Print Usage
    // -----------
    if (argc != 5) {
        cout << "Usage: " << argv[0] << " <filename_prefix>\n"
	     << "          <initial time step> <#time steps> <#files per time step>\n\n"
	     << "Merges <#files> gmv output files  from a pp3d++ run to one\n"
	     << "single gmv file.  The format of those files that are to be\n"
	     << "merged is expected to be\n"
	     << "  <filename_prefix>.t001.p000.gmv,\n"
	     << "  <filename_prefix>.t001.p001.gmv\n"
	     << "A file named <filename_prefix>.gmv is created. It contains\n"
	     << "besides all the information  from the input files material\n"
	     << "characteristics: the cells from different input files have\n"
	     << "different values for their material characteristics.\n\n"
	     << "A set of gmv output files that contain only grid information\n"
	     << "can be merged by setting the initial time step to -1 and the\n"
	     << "number of time steps to 1.  In this case the file pattern is\n"
	     << "exptected be\n"
	     << "  <filename_prefix>.p000.gmv,\n"
	     << "  <filename_prefix>.p001.gmv\n\n"
	     << "Version: " << VERSION << "\n\n";
        exit (0);
    }

    vInitialTimeStep = atoi(argv[2]);             // initial timestep
    vNumTimeSteps = atoi(argv[3]);                // number of timesteps
    vNumProcs     = atoi(argv[4]);                // number of input files per timestep
    vector< int > vNumElements(vNumProcs, 0);     // remember number of elements for each separate input file

    cout << argv[0] << " " << VERSION << '\n';
    for (int t=vInitialTimeStep;
	 abs(t - vInitialTimeStep) < abs(vNumTimeSteps);
	 t += vNumTimeSteps / abs(vNumTimeSteps) * 1) {
        // Re-initialize counters for each file
        vNumVerticesTotal = 0;       vNumElementsTotal = 0;
        vNumVerticesCurrentFile = 0; vNumVerticesSoFar = 0;
        vNumElementsCurrentFile = 0; vNumElementsSoFar = 0;

        // ----------------------------------------------------------
        // Open all files and sum the number of vertices and elements.
        // ----------------------------------------------------------
        for (int i=0; i < vNumProcs; i++) {
            infile = argv[1];

	    // Build up the current file name string
	    if (vInitialTimeStep > -1) {
		infile += ".t" ;
		if (t < 1000) {
		    infile += int_to_string (t, 3);
		} else {
		    infile += int_to_string (t, 4);
		}
	    }
            infile += ".p" ;
            infile += int_to_string (i, 3);
            infile += ".gmv";

	    // Tell the user what we're doing right now.
            cout << "Reading input file <" << infile << "> ... ";
	    cout.flush();

            // Open file
            std::ifstream *file = new std::ifstream(infile.c_str());
            if (! *file) {
                cerr << '\n' << argv[0] << ": "
		     << "Cannot open file <" << infile << ">\n"
		     << "Program aborted. Nothing done.\n\n";
                exit (FILE_NOT_FOUND);
            }

            // Determine whether file contains gmv input
            *file >> entry;
            if (entry != "gmvinput") {
                cerr << '\n' << argv[0] << ": "
                          << "File <" << infile << "> is not in gmv format.\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (NOT_GMV_FORMAT);
            }

	    // ----------------------------------------------------------
	    // Section to determine the number of vertices = nodes.
	    // ----------------------------------------------------------
            // Read node info.
            flag = false;
            while (*file >> entry) {
                if (entry == "nodes") {   flag = true; break;   }
            }

            if (flag != true) {
                cerr << '\n' << argv[0] << ": "
                          << "Could not find node information in file <" << infile << ">\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (NODE_INFO_NOT_FOUND);
            }

            // Read number of vertices
            *file >> entry;
            vNumVerticesCurrentFile = atoi(entry.c_str());
            vNumVerticesTotal      += atoi(entry.c_str());
	    // ----------------------------------------------------------
	    // End Section to determine the number of vertices = nodes.
	    // ----------------------------------------------------------


	    // ----------------------------------------------------------
	    // Section to determine the number of elements
	    // ----------------------------------------------------------
            // Read element info.
            flag = false;
	    while (*file >> entry) {
		if (entry == "cells") {   flag = true; break;   }
	    }

            if (flag != true) {
                cerr << '\n' << argv[0] << ": "
                          << "Could not find element information in file <" << infile << ">\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (ELEMENT_INFO_NOT_FOUND);
            }

            // Read number of elements
            *file >> entry;
            vNumElements[i]    = atoi(entry.c_str());
            vNumElementsTotal += atoi(entry.c_str());
	    // ----------------------------------------------------------
	    // End Section to determine the number of elements
	    // ----------------------------------------------------------


	    // ----------------------------------------------------------
	    // Section to determine whether what optional fields we have
	    // (pressure, temperature, vorticity_x, vorticity_y, 
	    //  vorticity_z, , vorticity_total)
	    // to allocate memory as needed.
	    // ----------------------------------------------------------

	    // Do this investigations only for the first file
	    if (i == 0) {
		// Skip "hex 8" (= 2 strings) and 8 nodes (in total 10 strings)
		// as many times as we have cells
		for (int j=0; j < vNumElements[i] * 10; j++, *file >> entry);

		while (file->good()) {
		    *file >> entry;

		    if (file->eof()) break;

		    // Check if we have velocity information now
		    if (entry == "velocity") {
			flagVelocity = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile * 3; j++, *file >> entry);
		    }

		    // Check if we have pressure information now
		    else if (entry == "pressure") {
			flagPressure = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile; j++, *file >> entry);
		    }

		    // Check if we have temperature information now
		    else if (entry == "temperature") {
			flagTemperature = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile; j++, *file >> entry);
		    }

		    // Check if we have vorticity_x information now
		    else if (entry == "vorticity_x") {
			flagVorticity_x = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile; j++, *file >> entry);
		    }

		    // Check if we have vorticity_y information now
		    else if (entry == "vorticity_y") {
			flagVorticity_y = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile; j++, *file >> entry);
		    }

		    // Check if we have vorticity_z information now
		    else if (entry == "vorticity_z") {
			flagVorticity_z = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile; j++, *file >> entry);
		    }

		    // Check if we have total vorticity information now
		    else if (entry == "vorticity_total") {
			flagVorticity_total = true;
			// Skip corresponding data that follows
			file->ignore(2);
			for (int j=0; j < vNumVerticesCurrentFile; j++, *file >> entry);
		    }

		    // Check if we have time information now
		    else if (entry == "probtime") {
			flagTime = true;
			// Skip corresponding data that follows
			file->ignore(10000, '\n');
		    }

		    else if (entry == "variable"  ||  entry == "endvars"  ||  entry == "endgmv") {
			     // Do nothing for this keyword
		    }
		    else {
			cout << "Unhandled case: " << entry << " found.\n";
		    }
		}
	    }

            // Close file.
            delete file;
            cout << "done.\n";
        } // end for loop over all input files

        // -----------------
        // Print statistics.
        // -----------------
	string message;
	int messageCounter = 0;
        cout << '\n'
                  << "Statistics:\n"
                  << "-----------\n"
                  << "Number of vertices: " << vNumVerticesTotal << '\n'
                  << "Number of elements: " << vNumElementsTotal << '\n';
	// Found whether velocity etc.
	message += "Found ";
	if (flagVelocity) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "velocity";
	}
	if (flagPressure) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "pressure";
	}
	if (flagTemperature) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "temperature";
	}
	if (flagVorticity_x) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "vorticity_x";
	}
	if (flagVorticity_y) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "vorticity_y";
	}
	if (flagVorticity_z) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "vorticity_z";
	}
	if (flagVorticity_total) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "vorticity_total";
	}
	if (flagTime) {
	    if (messageCounter) message += ", ";
	    messageCounter++;
	    message += "time stamp";
	}

	if (messageCounter) 
	    cout << message << " fields.\n";
	else
	    cout << message << "no data except nodes and elements.\n";
	cout << '\n';

        // ----------------
        // Allocate memory.
        // ----------------
        vector< vector<double> > vertex         (vNumVerticesTotal, vector<double> (3, -999.999));
        vector< vector<int>    > cell           (vNumElementsTotal, vector<int>    (8, -999));
        vector< vector<double> > velocity       (vNumVerticesTotal, vector<double> (3, -999.999));
        vector<double>           pressure       (vNumVerticesTotal, -999.999);
        vector<double>           temperature    (vNumVerticesTotal, -999.999);
        vector<double>           vorticity_x    (vNumVerticesTotal, -999.999);
        vector<double>           vorticity_y    (vNumVerticesTotal, -999.999);
        vector<double>           vorticity_z    (vNumVerticesTotal, -999.999);
        vector<double>           vorticity_total(vNumVerticesTotal, -999.999);
	double                   time;

        // -------------------------
        // Collect distributed data.
        // -------------------------
        for (int i=0; i < vNumProcs; i++) {
            infile = argv[1];
	    if (vInitialTimeStep > -1) {
		infile += ".t" ;
		if (t < 1000) {
		    infile += int_to_string (t, 3);
		} else {
		    infile += int_to_string (t, 4);
		}
	    }
            infile += ".p" ;
            infile += int_to_string (i, 3);
            infile += ".gmv";

            cout << "Processing input file <" << infile << "> ... ";
	    cout.flush();

            // Open file
            std::ifstream *file = new std::ifstream(infile.c_str());
            if (! *file) {
                cerr << '\n' << argv[0] << ": "
                          << "Cannot open file <" << infile << ">\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (FILE_NOT_FOUND);
            }

            // Determine whether file contains gmv input
            *file >> entry;
            if (entry != "gmvinput") {
                cerr << '\n' << argv[0] << ": "
                          << "File <" << infile << "> is not in gmv format.\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (NOT_GMV_FORMAT);
            }

            // Read node info.
            flag = false;
            while (*file >> entry) {
                if (entry == "nodes") {   flag = true; break;   }
            }

            if (flag != true) {
                cerr << '\n' << argv[0] << ": "
                          << "Could not find node information in file <" << infile << ">\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (NODE_INFO_NOT_FOUND);
            }

            // Read number of vertices and the coordinates of those vertices
            *file >> vNumVerticesCurrentFile;
            for (int i=0; i < vNumVerticesCurrentFile; i++)
                *file >> vertex[vNumVerticesSoFar + i][0];      // read x component of vertex i
            for (int i=0; i < vNumVerticesCurrentFile; i++)
                *file >> vertex[vNumVerticesSoFar + i][1];      // read y component of vertex i
            for (int i=0; i < vNumVerticesCurrentFile; i++)
                *file >> vertex[vNumVerticesSoFar + i][2];      // read z component of vertex i

            // Read cell info
            flag = false;
            while (*file >> entry) {
                if (entry == "cells") {   flag = true; break;   }
            }

            if (flag != true) {
                cerr << '\n' << argv[0] << ": "
                          << "Could not find cells information in file <" << infile << ">\n"
                          << "Program aborted. Nothing done.\n\n";
                exit (ELEMENT_INFO_NOT_FOUND);
            }

            // Read number of cells and those cells themselves
            *file >> vNumElementsCurrentFile;
            for (int i=0; i < vNumElementsCurrentFile; i++) {
                file->ignore(6);                      // Skip "hex 8"
                *file >> entry;                       // Edge 1
                cell[vNumElementsSoFar + i][0] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 2
                cell[vNumElementsSoFar + i][1] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 3
                cell[vNumElementsSoFar + i][2] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 4
                cell[vNumElementsSoFar + i][3] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 5
                cell[vNumElementsSoFar + i][4] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 6
                cell[vNumElementsSoFar + i][5] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 7
                cell[vNumElementsSoFar + i][6] = vNumVerticesSoFar + atoi(entry.c_str());
                *file >> entry;                       // Edge 8
                cell[vNumElementsSoFar + i][7] = vNumVerticesSoFar + atoi(entry.c_str());
            }

            // Read velocity info (if the first file of the sequence contained that)
	    if (flagVelocity) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "velocity") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find velocity information in file <" << infile << ">\n"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_VELOCITY_INFO);
		    }
		} else {
		    // Read velocity in each vertex
		    file->ignore(2);                      // Skip " 1" from "velocity 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> velocity[vNumVerticesSoFar + i][0];      // read x component of velocity in vertex i
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> velocity[vNumVerticesSoFar + i][1];      // read y component of velocity in vertex i
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> velocity[vNumVerticesSoFar + i][2];      // read z component of velocity in vertex i
		}
	    }

            // Read pressure info (if the first file of the sequence contained that)
	    if (flagPressure) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "pressure") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find pressure information in file <" << infile << ">\n"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_PRESSURE_INFO);
		    }
		} else {
		    // Read pressure in each vertex
		    file->ignore(2);                      // Skip " 1" from "pressure 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> pressure[vNumVerticesSoFar + i];       // read pressure in vertex i
		}
	    }

            // Read temperature info
	    if (flagTemperature) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "temperature") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find temperature information in file <" << infile << ">"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_TEMPERATURE_INFO);
		    }
		} else {
		    // Read temperature in each vertex
		    file->ignore(2);                      // Skip " 1" from "temperature 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> temperature[vNumVerticesSoFar + i];    // read temperature in vertex i
		}
	    }

            // Read vorticity_x info
	    if (flagVorticity_x) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "vorticity_x") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find vorticity_x information in file <" << infile << ">"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_VORTICITY_X_INFO);
		    }
		} else {
		    // Read vorticity_x in each vertex
		    file->ignore(2);                      // Skip " 1" from "vorticity_x 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> vorticity_x[vNumVerticesSoFar + i];    // read vorticity_x in vertex i
		}
	    }

            // Read vorticity_y info
	    if (flagVorticity_y) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "vorticity_y") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find vorticity_y information in file <" << infile << ">"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_VORTICITY_Y_INFO);
		    }
		} else {
		    // Read vorticity_y in each vertex
		    file->ignore(2);                      // Skip " 1" from "vorticity_y 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> vorticity_y[vNumVerticesSoFar + i];    // read vorticity_y in vertex i
		}
	    }

            // Read vorticity_z info
	    if (flagVorticity_z) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "vorticity_z") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find vorticity_z information in file <" << infile << ">"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_VORTICITY_Z_INFO);
		    }
		} else {
		    // Read vorticity_z in each vertex
		    file->ignore(2);                      // Skip " 1" from "vorticity_z 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> vorticity_z[vNumVerticesSoFar + i];    // read vorticity_z in vertex i
		}
	    }

            // Read vorticity_total info
	    if (flagVorticity_total) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "vorticity_total") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find vorticity_total information in file <" << infile << ">"
			     << "This was expected however because other files contained it.\n"
			     << "Program aborted. Nothing done.\n\n";
			exit(ONE_FILE_WITHOUT_VORTICITY_TOTAL_INFO);
		    }
		} else {
		    // Read vorticity_total in each vertex
		    file->ignore(2);                      // Skip " 1" from "vorticity_total 1"
		    for (int i=0; i < vNumVerticesCurrentFile; i++)
			*file >> vorticity_total[vNumVerticesSoFar + i];    // read vorticity_total in vertex i
		}
	    }

            // Read time variable
	    if (flagTime) {
		flag = false;
		while (*file >> entry) {
		    if (entry == "probtime") {   flag = true; break;   }
		}

		if (flag != true) {
		    if (vInitialTimeStep > -1) {
			cerr << "\n  " << argv[0] << ": "
			     << "Could not find time information in file <" << infile << ">"
			     << " Ignored.";
		    }
		} else {
		    *file >> time;
		}
	    }

            // Update log variables
            vNumVerticesSoFar += vNumVerticesCurrentFile;
            vNumElementsSoFar += vNumElementsCurrentFile;

            // Close file.
            delete file;
            cout << " done.\n";
        } // end for loop over all input files


        // ----------------------------
        // Write data into one big file
        // ----------------------------
        outfile = argv[1];
	if (vInitialTimeStep > -1) {
	    outfile += ".t" ;
	    if (t < 1000) {
		outfile += int_to_string (t, 3);
	    } else {
		outfile += int_to_string (t, 4);
	    }
	}
        outfile += ".gmv";

        cout << "Writing output file <" << outfile << "> ... ";
	cout.flush();

        // Open file
        std::ofstream *file = new std::ofstream(outfile.c_str());
        if (! *file) {
            cerr << '\n' << argv[0] << ": "
                      << "Cannot open file <" << outfile << ">\n"
                      << "Program aborted. Nothing done.\n\n";
            exit (FILE_NOT_FOUND);
        }

        // Write data to file
        *file << "gmvinput ascii\n";

        // Write node data
        *file << "nodes " << vNumVerticesTotal << "\n  ";
        for (vector< vector<double> >::const_iterator i = vertex.begin(); i < vertex.end(); ++i) {
            *file << i[0][0] << ' ';                  // Writing x coordinates
        }
        *file << "\n  ";
        for (vector< vector<double> >::const_iterator i = vertex.begin(); i < vertex.end(); ++i) {
            *file << i[0][1] << ' ';                  // Writing y coordinates
        }
        *file << "\n  ";
        for (vector< vector<double> >::const_iterator i = vertex.begin(); i < vertex.end(); ++i) {
            *file << i[0][2] << ' ';                  // Writing z coordinates
        }
        *file << '\n';

        // Write cell data
        *file << "cells " << vNumElementsTotal << '\n';
        for (vector< vector<int> >::const_iterator i = cell.begin(); i < cell.end(); ++i) {
            *file << "hex 8\n " 
                  << i[0][0] << ' ' << i[0][1] << ' ' << i[0][2] << ' ' << i[0][3] << ' '
                  << i[0][4] << ' ' << i[0][5] << ' ' << i[0][6] << ' ' << i[0][7] << '\n';
        }

        // Write material information
        // (We give different material characteristics for cells from different input files)
        *file << "material " << vNumProcs << " 0\n";
        for (int i=0; i < (signed int) vNumElements.size(); i++) {
            *file << "proc_" << int_to_string(i, 3) << '\n';
        }
        k = 1;                                        // Material no. 0 is ignored by GMV. Thus start with 1.
        for (vector< int >::const_iterator i = vNumElements.begin(); i < vNumElements.end(); ++i) {
            *file << "  ";
            for (int j=0; j < i[0]; j++) {
                *file << k << " ";
            }
            *file << '\n';
            k++;
        }

        if (flagVelocity == true) {
            // Write velocity data
            *file << "velocity 1\n  ";
            for (vector< vector<double> >::const_iterator i = velocity.begin(); i < velocity.end(); ++i) {
                *file << i[0][0] << ' ';                  // Writing x component of velocity in current vertex
            }
            *file << "\n  ";
            for (vector< vector<double> >::const_iterator i = velocity.begin(); i < velocity.end(); ++i) {
                *file << i[0][1] << ' ';                  // Writing y component of velocity in current vertex
            }
            *file << "\n  ";
            for (vector< vector<double> >::const_iterator i = velocity.begin(); i < velocity.end(); ++i) {
                *file << i[0][2] << ' ';                  // Writing z component of velocity in current vertex
            }
            *file << '\n';
        }

        *file << "variable\n";
        
        if (flagPressure == true) {
            // Write pressure data
            *file << "pressure 1\n  ";
            for (vector< double >::const_iterator i = pressure.begin(); i < pressure.end(); ++i) {
                *file << i[0] << ' ';                     // Writing pressure in current vertex
            }
            *file << '\n';
        }

        if (flagTemperature == true) {
            // Write temperature data
            *file << "temperature 1\n  ";
            for (vector< double >::const_iterator i = temperature.begin(); i < temperature.end(); ++i) {
                *file << i[0] << ' ';                     // Writing temperature in current vertex
            }
            *file << '\n';
        }

        if (flagVorticity_x == true) {
            // Write vorticity_x data
            *file << "vorticity_x 1\n  ";
            for (vector< double >::const_iterator i = vorticity_x.begin(); i < vorticity_x.end(); ++i) {
                *file << i[0] << ' ';                     // Writing vorticity_x in current vertex
            }
            *file << '\n';
        }

        if (flagVorticity_y == true) {
            // Write vorticity_y data
            *file << "vorticity_y 1\n  ";
            for (vector< double >::const_iterator i = vorticity_y.begin(); i < vorticity_y.end(); ++i) {
                *file << i[0] << ' ';                     // Writing vorticity_y in current vertex
            }
            *file << '\n';
        }

        if (flagVorticity_z == true) {
            // Write vorticity_z data
            *file << "vorticity_z 1\n  ";
            for (vector< double >::const_iterator i = vorticity_z.begin(); i < vorticity_z.end(); ++i) {
                *file << i[0] << ' ';                     // Writing vorticity_z in current vertex
            }
            *file << '\n';
        }

        if (flagVorticity_total == true) {
            // Write vorticity_total data
            *file << "vorticity_total 1\n  ";
            for (vector< double >::const_iterator i = vorticity_total.begin(); i < vorticity_total.end(); ++i) {
                *file << i[0] << ' ';                     // Writing vorticity_total in current vertex
            }
            *file << '\n';
        }

        *file << "endvars\n";

        if (flagTime == true) {
            // Write time data
            *file << "probtime " << time << '\n';
        }

        *file << "endgmv\n";

        // Close file
        delete file;
        cout << "done.\n";
    } // for each timestep

    return 0;
}

